/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 *                                                                          *
 * FILE NAME:   IAI_EZMac_Lite.h                                            *
 *                                                                          *
 * DESIGNER:	Lukacs, Miklos; Kovacs, Krisztian                            *
 * PROGRAMMER: Lukacs, Miklos                                               *
 *                                                                          *
 * DESCRIPTION: EZMac function definitions                                  *
 *                                                                          *
 * REVISION:    01_5  Maj  10, 2005     From the original EZmac version     *
 *                                      removed:                            *
 *                                     - Multicast addressing.              *
 *                                     - Network address mode.              *
 *                                     - Reduced address mode.              *
 *                                     - Dynamic packet length.             *
 *                                     - Error detection.                   *
 *                                     - CID is mandatory.                  *
 *              01_6  Aug    11, 2005                                       *
 *                                                                          *
 *				01_7  Dec	 08, 2005	Port to HiTech						*
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#pragma nolist

#ifndef EZmac_Lite
#define  EZmac_Lite


                /* ======================================= *
                 *         I N T R O D U C T I O N         *
                 * ======================================= */
/*
Using EZmac is only consist of calling the following functions:
char EZMacIdle (void);
char EZMac_Transmit (void);
char EZMac_Receive (void);
char EZMacReg_Write (MacRegs Name,char Value);
char EZMacReg_Read (MacRegs Name,char* Value);
char EZMacBuf_Write (char* buf);
char EZMacBuf_Read (char* buf);
Before any transmit or receive the RF parameters must be configured by EZMacReg_Write.
Before transmit, the data buffer must be filled with EZMacBuf_Write.
After receive, the received packet can be read by EZMacBuf_Read.

The MAC uses only two resources of the microcontroller, the Timer1 IT, and the external IT.
For the correct operation the following routines must be included into the global ISR:
(void) EZMac_TimerISR (void);
(void) EZMac_CommISR (void);

There is no any restriction about the structure of the application program, the only requirement is that:
this two routines must be included into the very first lines of main() function:
(void) EZMac_SysInit (void);
(void) EZMac_Config (void);

*/
                /* ======================================= *
                 *           P A R A M E T E R S           *
                 * ======================================= */


#define  PLEN_MAX           16

#define BAND915

#ifdef BAND315
#define  FREQ_Band         0x0000      // for IA4420 config command
//#define FREQ_C1   1
//#define FREQ_C2   31
//#define FREQ_start ...
//#define FREQ_step  ..
//#define FREQ_maxid ..

#elif defined(BAND433)
#define  FREQ_Band         0x0010      // for IA4420 config command
//#define FREQ_C1   1
//#define FREQ_C2   43
//#define FREQ_start ...
//#define FREQ_step  ..
//#define FREQ_maxid ..

#elif defined(BAND868)
#define  FREQ_Band         0x0020      // for IA4420 config command
//#define FREQ_C1   2
//#define FREQ_C2   43
//#define FREQ_start ...
//#define FREQ_step  ..
//#define FREQ_maxid ..

#elif defined(BAND915)
#define  FREQ_Band         0x0030      // for IA4420 config command
//#define FREQ_C1   3
//#define FREQ_C2   30
#define FREQ_start 112
#define FREQ_step  64
#define FREQ_maxid 60

#endif


                /* ======================================= *
                 *          D E F I N I T I O N S          *
                 * ======================================= */

//#define CCS_COMPILER

#define  MAC_OK             0
#define  STATE_ERROR        1
#define  NAME_ERROR         2
#define  VALUE_ERROR        3
#define  WKUP_ERROR         4

// Changed by Laszlo Kadar on 29.11.05
#ifdef HITECH_COMPILER
 _NEAR__ byte BANK_2 STRL,STRH;                 //holds the actual status of the EZRadio chip
#endif
#ifdef CCS_C_COMPILER
 byte STRL,STRH;                 //holds the actual status of the EZRadio chip
#endif

#ifdef FIX_MEM_ADDRESS
 #pragma byte STRL=0x29
 #pragma byte STRH=0x2A
#endif


#define DQD_bit    0x80
#define DRSSI_bit  0x01

//states of the EZMac
#define  EZMac_S_Sleep             0x00
#define  EZMac_S_WakeUp            0x80
#define  EZMac_S_Idle              0x40
#define  EZMac_S_RxDQDCheck        0x20
#define  EZMac_S_Receiving         0x24
#define  EZMac_S_PacketValid       0x28
#define  EZMac_S_TxCheckChannel    0x10
#define  EZMac_S_TxError           0x14
#define  EZMac_S_TxPacket          0x18

//the default addresses after POR
#define  DEFAULT_CID     0xCD
#define  DEFAULT_SFID    0xF1
#define  DEFAULT_SFLT    0xFE
#define  DEFAULT_SMSK    0xFF

                /* ======================================= *
                 *     T Y P E   D E C L A R A T I O N     *
                 * ======================================= */

typedef enum
{
    MCR,
    TCR,
    RCR,
    PFCR,
    F0,
    F1,
    F2,
    F3,
    F4,
    F5,
    F6,
    F7,
    FMASK,
    CID,
    SFLT,
    SMSK,
    SFID,
    RPL,
    MSR,
    RSR,
    R_CID,
    SID,
    DID,
    PLEN

}MacRegs;

#define EZ_LASTREG PLEN

#define  REGBANK_SIZE      18
#define  DATABANK_SIZE     6





                /* ======================================= *
                 *        M I S C E L L A N E O U S        *
                 * ======================================= */
#include "IAI_EZMac_Lite_Int.h"


                /* ======================================= *
                 *  F U N C T I O N   P R O T O T Y P E S  *
                 * ======================================= */

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMac_Wake_Up
  +
  + DESCRIPTION:    Switches the MAC from Sleep to Idle state. Puts the EZRadio chip into standby state
  +
  + RETURN:         Mac_OK: The operation was succesfull
  +
  +                 State_Error: The operation is ignored, because the current state of EZMAC is not Sleep.
  +
  + NOTES:          Switches on the oscillator circuit of the EZRadio chip.
  +                 This function returns immediately, but the transition from Sleep to Idle
  +                 state lasts 5ms. At the and of this period the Wake-up timer in the EZradio will be
  +                 switched on for the longest period (6300 day)
  +
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMac_Wake_Up (void);

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMac_Sleep
  +
  + DESCRIPTION:    Switches the MAC from Idle to Sleep state. Puts the EZRadio chip into completely halt state.
  +
  + RETURN:         Mac_OK: The operation was succesfull
  +                 State_Error: The operation is ignored, because the current state of EZMAC is not Idle.
  +
  + NOTES:          Every RF part of the EZRadio chip is switched off in this state,
  +                 even the oscillator circuit.
  +                 The Wake-up timer is also switched off, but the Low Battery detector state is not affected
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMac_Sleep (void);

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMacIdle
  +
  + DESCRIPTION:    Puts the RF MAC into idle state
  +
  + RETURN:         Mac_OK:The operation was succesfull
  +
  +                 State_Error:If an ongoing transmission, or reception was aborted (the current
  +                 state is Transmit packet or Receive packet)
  +
  +                 Wkup_Error:the current state is Sleep, or the wake-up process hasn't been completed.
  +
  + NOTES:          Puts the EZmac to idle state from any ongoing transmission, or reception state.
  +                 Switches off every part of the EZRadio chip, except the oscillator circuit.
  +                 The registers are not initialized except MSR.
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
// Changed by Laszlo Kadar on 26.11.05
#ifdef HITECH_COMPILER
 #pragma interrupt_level 1
#endif
char EZMac_Idle (void);

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMac_Transmit
  +
  + DESCRIPTION:    Transmits the buffer content through the RF interface
  +
  + RETURN:         Mac_OK:The transmission started succesfully
  +
  +                 State_Error:The transmission hasn't been started because the MAC is busy
  +                 (the current state is not Idle or Transmit Error)
  +
  +                 Wkup_Error:The transmission hasn't been started because the current state is
  +                 Sleep, or the wake-up process hasn't been completed.
  +
  + NOTES:          The RF parameters must be set before call this function.
  +                 Sends PLEN bytes, so PLEN must be set by EZMacReg_Write before call.
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMac_Transmit (void);

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME: EZMac_Receive
  +
  + DESCRIPTION:    This function is seeking for a valid transmission on the defined frequencies.
  +                 If a valid transmission is detected then receives the packet through the RF interface
  +
  + RETURN:         Mac_OK: Searching for a valid transmission has been started succesfully
  +
  +                 State_Error:The searching hasn't been started because the MAC is busy
  +                 (the current state is not Idle)
  +
  +                 Wkup_Error:The searching hasn't been started because the current state is
  +                 Sleep, or the wake-up process hasn't been completed.
  +
  + NOTES:          The RF parameters must be set before call this function.
  +                 The received data lenght can be read from PLEN by EZMacReg_Read.
  +                 This lenght shall be less or equal to PLEN_MAX.
  +
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMac_Receive (void);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME: EZMacReg_Write
  +
  + DESCRIPTION:    Writes 'value' to register identified by 'name'
  +
  + RETURN:         Mac_OK:The operation was succesfull
  +
  +                 Name_Error: The register name is invalid
  +
  +                 Value_error: The value parameter is out of the range of register 'Name'
  +
  +                 State_Error: The operation is ignored, because reception or transmission is ongoing.
  +
  + NOTES:          Examines the Name, and Value.
  +                 PLEN <= PLEN_MAX
  +                 etc.
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMacReg_Write (MacRegs Name,char Value);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME: EZMacReg_Read
  +
  + DESCRIPTION:    Reads the content of register identified by 'name' and stores it to *Value
  +
  + RETURN:         Mac_OK:The operation was succesfull
  +
  +                 Name_Error: The register name is invalid
  +
  + NOTES:          none.
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMacReg_Read (MacRegs Name,char* Value);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMacBuf_Write
  +
  + DESCRIPTION:    Copies PLEN number of bytes from 'buf' to MAC data buffer
  +
  + RETURN:         Mac_OK:The operation was succesfull
  +
  +                 State_Error: The operation is ignored, because reception or transmission is ongoing.
  +
  + NOTES:          PLEN must be set before call this function.
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMacBuf_Write (char* buf);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMacBuf_Read
  +
  + DESCRIPTION:    Copies PLEN number of bytes from MAC data buffer to '*buf'
  +
  + RETURN:         Mac_OK:The operation was succesfull
  +
  +                 State_Error: The operation is ignored, because reception is ongoing.
  +
  + NOTES:          If the packet lenght is not fixed, then PLEN is coming from
  +                 the received packet, less or equal to PLEN_MAX.
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZMacBuf_Read (char* buf);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMac_TimerISR
  +
  + DESCRIPTION:    Timer Interrupt Service Routine.
  +
  + RETURN:         none.
  +
  + NOTES:          EZmac uses the Timer1 resource of the microcontroller.
  +                 This function call must be included into the global ISR.
  +
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void EZMac_TimerISR (void);



/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMac_CommISR
  +
  + DESCRIPTION:    External Interrupt Service Routine, handles the RF communication.
  +
  + RETURN:         none.
  +
  + NOTES:          EZmac uses the External interrupt resource of the microcontroller.
  +                 This function call must be included into the global ISR.
  +
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void EZMac_CommISR (void);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMac_SysInit
  +
  + DESCRIPTION:    Microcontroller initialization routine.
  +
  + RETURN:         none.
  +
  + NOTES:          Configures the microcontroller resources whose are needed by EZmac.
  +                 This function call must be included into the main() function of the code,
  +                 prior to any other uses of the MAC.
  +
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
void EZMac_SysInit (void);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZMac_Config
  +
  + DESCRIPTION:    EZmac initialization routine.
  +
  + RETURN:         none.
  +
  + NOTES:          Initializes EZmac registers, and the RF communication chip.
  +                 This function call must be included into the main() function of the code,
  +                 prior to any other uses of the MAC, and after EZMac_SysInit
  +
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
void EZMac_Config (void);


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZRadio_Write
  +
  + DESCRIPTION:    Sends EZcommand to the EZRadio chip via the SPI interface.
  +
  + RETURN:         Mac_OK: The operation was succesfull

  +                 State_Error: The operation is ignored, because the current
  +                 state of EZMAC is not Idle or Sleep.
  +
  + NOTES:          The MAC must be in Sleep or Idle state when calling this function
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
// Changed by Laszlo Kadar on 26.11.05
#ifdef HITECH_COMPILER
 #pragma interrupt_level 1
#endif
char EZRadio_Write (int16 EZcommand);

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +
  + FUNCTION NAME:  EZRadio_Statusread
  +
  + DESCRIPTION:    Reads the Status of the EZRadio chip via the SPI interface
  +                 and stores it to STRL and STRH global variables.
  +
  + RETURN:         Mac_OK: The operation was succesfull

  +                 State_Error: The operation is ignored, because the current
  +                 state of EZMAC is not Idle or Sleep.
  +
  + NOTES:          The MAC must be in Sleep or Idle state when calling this function
  +                 The MAC updates regularly the STRH global variable which is reflecting
  +                 the Interrupt status of the EZRadio chip.
  +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#pragma separate
char EZRadio_Statusread (void);

#endif
#pragma list
